/*_############################################################################
  _## 
  _##  subagent_win32.cpp  
  _## 
  _##
  _##  AGENT++Win32 API Version 1.4a
  _##  ---------------------------------------------------------
  _##  Copyright (C) 2003-2004, Frank Fock, All rights reserved.
  _##  
  _##  LICENSE AGREEMENT
  _##
  _##
  _##  Use of this software is subject to the license agreement you received
  _##  with this software and which can be downloaded from 
  _##  http://www.agentpp.com/agentX++/license.txt
  _##
  _##  This is licensed software and may not be used in a commercial
  _##  environment, except for evaluation purposes, unless a valid
  _##  license has been purchased.
  _##
  _##
  _##  Stuttgart, Germany, Thu Sep  2 00:08:13 CEST 2010 
  _##  
  _##########################################################################*/


// subagent_win32.cpp : Defines the entry point for the DLL application.
//

#include <snmp.h>

#include "stdafx.h"
#include "subagent_win32.h"
#include <snmp_pp/log.h>
#include <agent_pp/system_group.h>

BOOL APIENTRY DllMain( HANDLE hModule, 
                       DWORD  ul_reason_for_call, 
                       LPVOID lpReserved
					 )
{
	switch (ul_reason_for_call)
	{
	case DLL_PROCESS_ATTACH:
	case DLL_THREAD_ATTACH:
	case DLL_THREAD_DETACH:
	case DLL_PROCESS_DETACH:
		break;
	}
    return TRUE;
}

AgentLog& Win32Log::operator +=(const LogEntry* log) 
{
	INT logLevel = SNMP_LOG_VERBOSE;
	switch (log->get_class()) {
		case ERROR_LOG:
			if (log->get_level() == 0)
				logLevel = SNMP_LOG_FATAL;
			else
				logLevel = SNMP_LOG_ERROR;
			break;
		case WARNING_LOG:
			logLevel = SNMP_LOG_WARNING;
			break;
		case INFO_LOG:
			logLevel = SNMP_LOG_TRACE;
			break;
		case DEBUG_LOG:
			logLevel = SNMP_LOG_VERBOSE;
			break;
	}
	SnmpUtilDbgPrint(logLevel, "%s\n", log->get_value());
	return *this;
}

#ifdef _SNMPv3
Win32Request::Win32Request(const Pdux& pdu): Request(pdu, UTarget())
#else
Win32Request::Win32Request(const Pdux& pdu): Request(pdu, CTarget())
#endif
{
	target.set_version(version2c);
}

Win32Request::~Win32Request() 
{
}

Oidx Win32RequestList::genericTrapOid("1.3.6.1.6.3.1.1.5");

Win32RequestList::Win32RequestList(): RequestList()
{
	num_regions_registered = 0;
	num_regions = 0;
	regions = 0;
	trapEvent = 0;
	enterpriseOid = 0;
#ifdef _SNMPv3
	set_vacm(new Win32NoVacm());
#endif
}

Win32RequestList::~Win32RequestList() 
{
	CloseHandle(trapEvent);
	SnmpUtilMemFree(regions);
	if (enterpriseOid) {
		SnmpUtilOidFree(enterpriseOid);
		enterpriseOid = 0;
	}
#ifdef _SNMPv3
	delete get_vacm();
#endif
}

BOOL Win32RequestList::init(Mib* m, HANDLE *phSubagentTrapEvent, 
						    AsnObjectIdentifier *pFirstSupportedRegion)
{
	mib = m;
	/* Create an event to indicate that a trap needs to be sent */
	SnmpSvcSetLogLevel(SNMP_LOG_VERBOSE);
	SnmpSvcSetLogType(SNMP_OUTPUT_TO_LOGFILE);

    *phSubagentTrapEvent = CreateEvent(NULL, FALSE, FALSE, NULL);
    if (*phSubagentTrapEvent != NULL) {
		trapEvent = *phSubagentTrapEvent;
	}
	if (!mib) {
		LOG_BEGIN(ERROR_LOG | 1);
		LOG("Win32RequestList: Mib not set on initialization!");
		LOG_END;
		return FALSE;
	}
	MibContext* ctx = mib->get_default_context();
	int sz = ctx->get_num_groups();
	if (sz == 0) {
		LOG_BEGIN(ERROR_LOG | 1);
		LOG("Win32RequestList: No MibGroups added to Mib");
		LOG_END;
		return FALSE;
	}
	// allocate memory for regions 
	regions = 
		(AsnObjectIdentifier*)SnmpUtilMemAlloc(sz * sizeof(AsnObjectIdentifier));
	if (!regions)
		return FALSE;
	num_regions = sz;
	num_regions_registered = 1;

	// assign first region (probably last)
	OidListCursor<MibGroup> cur(ctx->get_groups());
	for (int i=0; (i<sz) && (cur.get()); cur.next(),i++) {
		copy_oid(*cur.get()->key(), regions[i]);
	}
	*pFirstSupportedRegion = regions[0];

	// assign system up time
	sysUpTime::start = SnmpSvcGetUptime();
	return TRUE;
}

BOOL Win32RequestList::init_next(AsnObjectIdentifier *pNextSupportedRegion)
{
	if (num_regions_registered < num_regions) {
		*pNextSupportedRegion = regions[num_regions_registered++];
		return TRUE;
	}
	return FALSE;
}

AsnInteger32 Win32RequestList::get_generic_id(const Oidx& oid) const
{
	if ((oid.len() == 10) && (oid.in_subtree_of(genericTrapOid)))
		return oid[9];
	return 6;
}

AsnInteger32 Win32RequestList::get_specific_id(const Oidx& oid) const
{
	if ((oid.len() == 0) || 
		((oid.len() == 10) && (oid.in_subtree_of(genericTrapOid))))
		return 0;
	return oid[oid.len()-1];
}


BOOL Win32RequestList::query(DWORD dwRequestType,   // extension agent request type
							 DWORD dwTransactionId, // identifier of the incoming PDU
						     SnmpVarBindList *pVarBindList, // pointer to variable binding list
							 AsnOctetString *pContextInfo,  // pointer to context information
							 AsnInteger32 *pErrorStatus,    // pointer to SNMPv2 error status
							 AsnInteger32 *pErrorIndex      // pointer to the error index
							 ) 
{
	LOG_BEGIN(DEBUG_LOG | 2);
	LOG("Win32RequestList: processing query (type)(tid)(sz)");
	LOG(dwRequestType);
	LOG(dwTransactionId);
	LOG(pVarBindList->len);
	LOG_END;

	Win32Request* req = 0;
	if ((dwRequestType == SNMP_EXTENSION_SET_COMMIT) ||
		(dwRequestType == SNMP_EXTENSION_SET_UNDO) ||
		(dwRequestType == SNMP_EXTENSION_SET_CLEANUP)) {
		req = (Win32Request*)get_request(dwTransactionId);
	}
	else {
		Pdux pdu;
		pdu.set_request_id(dwTransactionId);

		switch (dwRequestType) {
			case SNMP_EXTENSION_SET_TEST: {
				for (unsigned int i=0; i<pVarBindList->len; i++) {
					Vbx vb(Oid
						((unsigned long *) pVarBindList->list[i].name.ids, 
						pVarBindList->list[i].name.idLength));
					AsnAny* any = &pVarBindList->list[i].value;
					switch (any->asnType) {
						//case ASN_INTEGER:
						case ASN_INTEGER32: {
							vb.set_value(SnmpInt32(any->asnValue.number));
							break;
						}
						/* undistinguishable from Gauge32
						case ASN_UNSIGNED32: {
							vb.set_value(SnmpUInt32(any->asnValue.unsigned32));
							break;
						}
						*/
						case ASN_COUNTER64: {
							vb.set_value(Counter64(any->asnValue.counter64.HighPart,
								any->asnValue.counter64.LowPart));
							break;
						}
						case ASN_OCTETSTRING: {
							vb.set_value(OctetStr(any->asnValue.string.stream,
								any->asnValue.string.length));
							break;
						}
						case ASN_OPAQUE: {
							vb.set_value(OpaqueStr(any->asnValue.string.stream,
								any->asnValue.string.length));
							break;
						}
						case ASN_OBJECTIDENTIFIER: {
							vb.set_value(Oid((const unsigned long *) any->asnValue.object.ids,
								any->asnValue.object.idLength));
							break;
						}
						case ASN_IPADDRESS: {
							Oid oidIP;
							for (unsigned int i=0; i< any->asnValue.address.length; i++) {
								oidIP += any->asnValue.address.stream[i];
							}
							vb.set_value(IpAddress( oidIP.get_printable()));
							break;
						}
						case ASN_COUNTER32: {
							vb.set_value(Counter32(any->asnValue.counter));
							break;
						}
						case ASN_GAUGE32: {
							vb.set_value(Gauge32(any->asnValue.gauge));
							break;
						}
						case ASN_TIMETICKS: {
							vb.set_value(TimeTicks(any->asnValue.ticks));
							break;
						}
						default:
							return FALSE;
					}
					pdu += vb;
				}
				break;
			}
			default: {
				for (unsigned int i=0; i<pVarBindList->len; i++) {
					Vbx vb(Oid
						((unsigned long *) pVarBindList->list[i].name.ids, 
						pVarBindList->list[i].name.idLength));
					pdu += vb;
				}
				break;
			}
		}
		switch (dwRequestType) {
			case SNMP_EXTENSION_GET: 
			{
				pdu.set_type(sNMP_PDU_GET);
				break;
			}
			case SNMP_EXTENSION_GET_NEXT: 
			{
				pdu.set_type(sNMP_PDU_GETNEXT);
				break;
			}
			case SNMP_EXTENSION_SET_TEST:
			{
				pdu.set_type(sNMP_PDU_SET);
				break;
			}	
			default:
				return 0;
		}
		req = new Win32Request(pdu);
		req->set_transaction_id(dwTransactionId);
		add_request(req);
	}
	if (req) {
		int status = SNMP_ERROR_SUCCESS;
		switch(dwRequestType) {
			case SNMP_EXTENSION_SET_TEST:
				status = mib->process_prepare_set_request(req);
				break;
			case SNMP_EXTENSION_SET_COMMIT:
				status = mib->process_commit_set_request(req);
				break;
			case SNMP_EXTENSION_SET_CLEANUP:
				mib->process_cleanup_set_request(req);
				answer(req);
				break;
			case SNMP_EXTENSION_SET_UNDO:
				status = mib->process_undo_set_request(req);
				answer(req);
				break;
			default:
				mib->do_process_request(req);
		}
		// copy result
		*pErrorIndex = req->get_error_index();
		*pErrorStatus = req->get_error_status();
		// copy variable bindings
		if ((dwRequestType != SNMP_EXTENSION_SET_TEST) &&
			(dwRequestType != SNMP_EXTENSION_SET_CLEANUP) &&
			(dwRequestType != SNMP_EXTENSION_SET_COMMIT) &&
			(dwRequestType != SNMP_EXTENSION_SET_UNDO)) {
			Pdux* pdu = req->get_pdu();
			int sz = pdu->get_vb_count();
			Vbx* vbs = new Vbx[sz];
			pdu->get_vblist(vbs, sz);
			copy_vbs(vbs, sz, pVarBindList);
			delete[] vbs;
			// done
			delete req;
		}
		else if ((dwRequestType == SNMP_EXTENSION_SET_UNDO) ||
				 (dwRequestType == SNMP_EXTENSION_SET_CLEANUP))
			delete req;
		return (status == SNMP_ERROR_SUCCESS);
	}
	return FALSE;
}

BOOL Win32RequestList::trap(AsnObjectIdentifier *pEnterpriseOid, 
	                        AsnInteger32 *pGenericTrapId,
                            AsnInteger32 *pSpecificTrapId,
                            AsnTimeticks *pTimeStamp,
                            SnmpVarBindList *pVarBindList)
{
	
#ifdef _THREADS
	trapEventLock.lock();
#endif
	if (enterpriseOid) {
		SnmpUtilOidFree(enterpriseOid);
		enterpriseOid = 0;
	}

	Pdux* pdu = trapEventQueue.removeFirst();
	if (!pdu)
		return FALSE;

	enterpriseOid = 
		(AsnObjectIdentifier*)SnmpUtilMemAlloc(sizeof(AsnObjectIdentifier));
	Oidx oid;
	pdu->get_notify_enterprise(oid);
	copy_oid(oid, *enterpriseOid);
	*pEnterpriseOid = *enterpriseOid;

	pdu->get_notify_id(oid);
	*pGenericTrapId = get_generic_id(oid);
	*pSpecificTrapId = get_specific_id(oid);
	*pTimeStamp = SnmpSvcGetUptime();

	int sz = pdu->get_vb_count();
	Vbx* vbs = new Vbx[sz];
	pdu->get_vblist(vbs, sz);
	pVarBindList->len = sz;
	if (sz == 0)
		pVarBindList->list = 0;
	else
		pVarBindList->list = (SnmpVarBind*)
			SnmpUtilMemAlloc(sz * sizeof(SnmpVarBind));
	copy_vbs(vbs, sz, pVarBindList);
	delete[] vbs;
	delete pdu;

	return TRUE;
}


void Win32RequestList::copy_oid(const Oidx& in, AsnObjectIdentifier& out)
{
	out.idLength = in.len();
	out.ids = (UINT*)SnmpUtilMemAlloc(sizeof(UINT) * out.idLength);
	for (unsigned int i=0; i<out.idLength; i++) 
		out.ids[i] = in[i];
}

int Win32RequestList::copy_vbs(Vbx* vbs, unsigned int sz, 
							   SnmpVarBindList* vbList) const
{
	if (!vbList)
		return -1;
	vbList->len = sz;
	for (unsigned int i=0; i<vbList->len; i++) {
		Oidx oid;
		vbs[i].get_oid(oid);

		vbList->list[i].name.idLength = oid.len();
		vbList->list[i].name.ids = 
			(UINT*) SnmpUtilMemAlloc(vbList->list[i].name.idLength * 
									 sizeof(UINT));
		if (!vbList->list[i].name.ids) {
			return i;
		}
		memcpy(vbList->list[i].name.ids, 
			   oid.oidval()->ptr, 
			   vbList->list[i].name.idLength * sizeof(UINT));

		AsnAny* pAsn = &vbList->list[i].value;

		switch (vbs[i].get_syntax()) {
			case sNMP_SYNTAX_INT32: {
				pAsn->asnType = ASN_INTEGER32;
				vbs[i].get_value(pAsn->asnValue.number);
				break;
			}
			/* undistinguishable from GAUGE32
			case sNMP_SYNTAX_UINT32: {
				pAsn->asnType = ASN_UNSIGNED32;
				vbs[i].get_value(pAsn->asnValue.unsigned32);
				break;
			}
			*/
			case sNMP_SYNTAX_CNTR64: {
				Counter64 ctr;
				vbs[i].get_value(ctr);

				pAsn->asnType = ASN_COUNTER64;
				pAsn->asnValue.counter64.HighPart = ctr.high();
				pAsn->asnValue.counter64.LowPart  = ctr.low();
				break;
			}
			case sNMP_SYNTAX_OPAQUE: {
				OctetStr str;
				vbs[i].get_value(str);

				pAsn->asnType = ASN_OPAQUE;
				pAsn->asnValue.string.length = str.len();
				pAsn->asnValue.string.stream = (BYTE*) SnmpUtilMemAlloc(
//					pAsn->asnValue.string.stream, 
					pAsn->asnValue.string.length * sizeof(BYTE));
				if (pAsn->asnValue.string.stream == 0) {
					return i;
				}
				memcpy(pAsn->asnValue.string.stream, 
					str.data(), 
					pAsn->asnValue.string.length * sizeof(BYTE));
				pAsn->asnValue.string.dynamic = TRUE;
				break;
			}
			case sNMP_SYNTAX_OCTETS: {
				OctetStr str;
				vbs[i].get_value(str);

				pAsn->asnType = ASN_OCTETSTRING;
				pAsn->asnValue.string.length = str.len();
				pAsn->asnValue.string.stream = (BYTE*) SnmpUtilMemAlloc(
//					pAsn->asnValue.string.stream, 
					pAsn->asnValue.string.length * sizeof(BYTE));
				if (pAsn->asnValue.string.stream == 0) {
					return i;
				}
				memcpy(pAsn->asnValue.string.stream, 
					str.data(), 
					pAsn->asnValue.string.length * sizeof(BYTE));
				pAsn->asnValue.string.dynamic = TRUE;
				break;
			}
			case sNMP_SYNTAX_OID: {
				Oid dataOid;
				vbs[i].get_value(dataOid);

				pAsn->asnType = ASN_OBJECTIDENTIFIER;
				pAsn->asnValue.object.idLength = dataOid.len();
				pAsn->asnValue.object.ids = (UINT*) SnmpUtilMemAlloc(
					//pAsn->asnValue.object.ids, 
					pAsn->asnValue.object.idLength * sizeof(UINT));
				if (pAsn->asnValue.object.ids == 0) {
					return i;
				}
				memcpy(pAsn->asnValue.object.ids, 
					dataOid.oidval()->ptr, 
					pAsn->asnValue.object.idLength * sizeof(UINT));
				break;
			}
			case sNMP_SYNTAX_IPADDR: {
				const int IPADDRESS_LEN = 4;

				IpAddress addr;
				vbs[i].get_value(addr);

				pAsn->asnType = ASN_IPADDRESS;
				pAsn->asnValue.address.length = IPADDRESS_LEN;
				pAsn->asnValue.address.stream = (BYTE*) SnmpUtilMemAlloc(
					//pAsn->asnValue.address.stream, 
					pAsn->asnValue.address.length);
				if (pAsn->asnValue.address.stream == 0) {
					return 0;
				}
				for (int k=0; k<IPADDRESS_LEN; k++)
					pAsn->asnValue.address.stream[k] = addr[k];
				pAsn->asnValue.address.dynamic = TRUE;
				break;
			}
			case sNMP_SYNTAX_CNTR32: {
				pAsn->asnType = ASN_COUNTER32;
				vbs[i].get_value(pAsn->asnValue.counter);
				break;
			}
			case sNMP_SYNTAX_GAUGE32: {
				pAsn->asnType = ASN_GAUGE32;
				vbs[i].get_value(pAsn->asnValue.gauge);
				break;
			}
			case sNMP_SYNTAX_TIMETICKS: {
				pAsn->asnType = ASN_TIMETICKS;
				vbs[i].get_value(pAsn->asnValue.ticks);
				break;
			}
			case sNMP_SYNTAX_ENDOFMIBVIEW:
			case sNMP_SYNTAX_NOSUCHINSTANCE:
			case sNMP_SYNTAX_NOSUCHOBJECT:
			case sNMP_SYNTAX_NULL: {
				pAsn->asnType = vbs[i].get_syntax();
				pAsn->asnValue.number = 0;
				break;
			}
			default:
				return i;
		}
	}
	return 0;
}



int Win32RequestList::notify(const OctetStr&, const Oidx& oid, 
							 Vbx* vbs, int sz, 
							 unsigned int timestamp)
{
	Pdux* pdu = new Pdux(vbs, sz);
	pdu->set_notify_id(oid);
	pdu->set_notify_timestamp(timestamp);
#ifdef _THREADS
	trapEventLock.lock();
#endif
	trapEventQueue.addLast(pdu);
#ifdef _THREADS
	trapEventLock.unlock();
#endif
	SetEvent(trapEvent);
	return SNMP_ERROR_SUCCESS;
}

void Win32RequestList::answer(Request*  req) 
{
	Pdux* pdu = req->get_pdu();

	LOG_BEGIN(EVENT_LOG | 2);
	LOG("Win32RequestList: request answered (rid)(tid)(to)(err)(sz)");
	LOG(pdu->get_request_id());
	LOG(req->get_transaction_id());
	LOG(pdu->get_error_status());
	LOG(req->get_pdu()->get_vb_count());
	LOG_END;

	// do not delete req here!
	requests->remove(req);
}

Request* Win32RequestList::add_request(Request* req) 
{
	requests->add(req);
	return req;	
}


