/*_############################################################################
  _## 
  _##  agentx_pp_master.cpp  
  _## 
  _##
  _##  AGENT++Win32 API Version 1.4a
  _##  ---------------------------------------------------------
  _##  Copyright (C) 2003-2004, Frank Fock, All rights reserved.
  _##  
  _##  LICENSE AGREEMENT
  _##
  _##
  _##  Use of this software is subject to the license agreement you received
  _##  with this software and which can be downloaded from 
  _##  http://www.agentpp.com/agentX++/license.txt
  _##
  _##  This is licensed software and may not be used in a commercial
  _##  environment, except for evaluation purposes, unless a valid
  _##  license has been purchased.
  _##
  _##
  _##  Stuttgart, Germany, Thu Sep  2 00:08:13 CEST 2010 
  _##  
  _##########################################################################*/
// agentx_pp_master.cpp : Defines the entry point for the DLL application.
//

#include "stdafx.h"
#include "snmp.h"
#include <stdlib.h>
#include "agentx_pp_master.h"
#include <agent_pp/snmp_group.h>
#include <agent_pp/system_group.h>
#include <agent_pp/snmp_notification_mib.h>
#include <agent_pp/snmp_target_mib.h>
#include <agent_pp/snmp_community_mib.h>
#include <agentx_pp/agentpp_agentx_mib.h>
#include <agent_pp/agentpp_config_mib.h>
#include <agent_pp/v3_mib.h>
#include <agent_pp/notification_log_mib.h>
#include <snmp_pp/mp_v3.h>
#include <snmp_pp/oid_def.h>
#include <snmp_pp/log.h>

#ifdef AGENTPP_NAMESPACE
namespace Agentpp {
#endif


int MasterAgentXMibWin32::runStatus = 2;
HANDLE MasterAgentXMibWin32::hServiceMainThread = NULL;

VOID WINAPI ServiceMain(DWORD dwArgc, LPTSTR *argv)
{
	MasterAgentXMibWin32::Service(dwArgc, argv);
}

SERVICE_TABLE_ENTRY   serviceTableEntries[] = 
{ 
	{"SNMP", ServiceMain}, 
	{NULL, NULL}
};


boolean MasterAgentXMibWin32::set_service_status(
		DWORD status,
		DWORD checkPoint,
		DWORD waitHint)
{
    serviceStatus.dwCurrentState       = status; 
    serviceStatus.dwCheckPoint         = checkPoint; 
    serviceStatus.dwWaitHint           = waitHint;  
    if (!SetServiceStatus(hServiceStatusHandle, &serviceStatus)) { 
		long nError = GetLastError();
		LOG_BEGIN(ERROR_LOG | 1);
        LOG("SetServiceStatus failed (error)");
		LOG(nError); 
		LOG_END;
		return FALSE;
    } 
	return TRUE;
}

DWORD WINAPI HandlerEx(
  DWORD dwControl,     // requested control code
  DWORD dwEventType,   // event type
  LPVOID lpEventData,  // event data
  LPVOID lpContext     // user-defined context data
)
{
	/* Post status to a completion port to avoid tying up the primary thread */
	if ((MasterAgentXMibWin32::instance) 
			&& ((MasterAgentXMibWin32*)MasterAgentXMibWin32::instance)->hIOCPort) {
		PostQueuedCompletionStatus(
				((MasterAgentXMibWin32*)MasterAgentXMibWin32::instance)->hIOCPort,
				0, dwControl, NULL);
	}
	return NO_ERROR;
}

boolean MasterAgentXMibWin32::init_service() 
{
	DWORD   status = 0; 
    DWORD   specificError = 0xfffffff; 
 
    serviceStatus.dwServiceType        = SERVICE_WIN32; 
    serviceStatus.dwCurrentState       = SERVICE_START_PENDING; 
    serviceStatus.dwControlsAccepted   = SERVICE_ACCEPT_STOP | SERVICE_ACCEPT_SHUTDOWN | SERVICE_ACCEPT_PAUSE_CONTINUE; 
    serviceStatus.dwWin32ExitCode      = 0; 
    serviceStatus.dwServiceSpecificExitCode = 0; 
    serviceStatus.dwCheckPoint         = 0; 
    serviceStatus.dwWaitHint           = 0; 

	hIOCPort = CreateIoCompletionPort(INVALID_HANDLE_VALUE, NULL, NULL, 0);
	if (hIOCPort == 0) {
		long nError = GetLastError();
		LOG_BEGIN(WARNING_LOG | 1);
		LOG("CreateIoCompletinPort failed (error)");
		LOG(nError);
		LOG_END;
		return FALSE;
	}

	hServiceStatusHandle = RegisterServiceCtrlHandlerEx(serviceTableEntries[0].lpServiceName, HandlerEx, 0);
	if (hServiceStatusHandle == 0) {
		long nError = GetLastError();
		LOG_BEGIN(WARNING_LOG | 1);
		LOG("RegisterServiceCtrlHandler failed (error)");
		LOG(nError); 
		LOG_END;
		CloseHandle(hIOCPort);
		hIOCPort = NULL;
        return FALSE; 
    } 
	
    // Handle error 
	status = GetLastError(); 
    if (status != NO_ERROR) { 
        serviceStatus.dwCurrentState       = SERVICE_STOPPED; 
        serviceStatus.dwCheckPoint         = 0; 
        serviceStatus.dwWaitHint           = 0; 
        serviceStatus.dwWin32ExitCode      = status; 
        serviceStatus.dwServiceSpecificExitCode = specificError; 
        SetServiceStatus(hServiceStatusHandle, &serviceStatus); 
        return FALSE; 
    } 
	return TRUE;
}


void MasterAgentXMibWin32::Service(int argc, LPTSTR* argv)
{
	/* duplicate handle of this thread */
	if (!DuplicateHandle(GetCurrentProcess(), GetCurrentThread(),
			GetCurrentProcess(), &MasterAgentXMibWin32::hServiceMainThread,
			0, 0, DUPLICATE_SAME_ACCESS)) {
		MasterAgentXMibWin32::hServiceMainThread = NULL;
	}

	u_short port;
	MasterAgentXMibWin32* mib = new MasterAgentXMibWin32();
	runStatus = 2;

	boolean service = mib->service = mib->init_service();

	port = 161;
	for (int i=1; i<argc-1; i++) {
		if (_stricmp("-p", argv[i]) == 0) {
			port = atoi((const char*)argv[i+1]);
		}
	}

#ifndef _NO_LOGGING
	DefaultLog::log()->set_filter(ERROR_LOG, 4);
	DefaultLog::log()->set_filter(WARNING_LOG, 0);
	DefaultLog::log()->set_filter(EVENT_LOG, 1);
	DefaultLog::log()->set_filter(INFO_LOG, 1);
	DefaultLog::log()->set_filter(DEBUG_LOG, 0);
#endif
	Snmp::socket_startup();
	int status;
	// create snmp object here, to be destroyed before service is stopped
	Snmpx *snmp = new Snmpx(status, port);
	mib->snmp = snmp;

	if (status == SNMP_CLASS_SUCCESS) {

		LOG_BEGIN(EVENT_LOG | 1);
		LOG("main: SNMP listen port");
		LOG(port);
		LOG_END;
	}
	else {
		LOG_BEGIN(ERROR_LOG | 0);
		LOG("main: SNMP port init failed");
		LOG(status);
		LOG_END;
		exit(1);
	}

	unsigned int snmpEngineBoots = 0;
    OctetStr engineId(SnmpEngineID::create_engine_id(port));

    // you may use your own methods to load/store this counter
    status = mib->get_boot_counter(engineId, snmpEngineBoots);
    if ((status != SNMPv3_OK) && (status < SNMPv3_FILEOPEN_ERROR)) {
		LOG_BEGIN(ERROR_LOG | 0);
		LOG("main: Error loading snmpEngineBoots counter (status)");
		LOG(status);
		LOG_END;
		exit(1);
	}
	snmpEngineBoots++;
    status = mib->set_boot_counter(engineId, snmpEngineBoots);
    if (status != SNMPv3_OK) {
		LOG_BEGIN(ERROR_LOG | 0);
		LOG("main: Error saving snmpEngineBoots counter (status)");
		LOG(status);
		LOG_END;
		exit(1);
	}
	int stat;
    v3MP *v3mp = new v3MP(engineId, snmpEngineBoots, stat);
    mib->v3mp = v3mp;

	// retrieve some values from mib object so they can be used
	// after mib object is deleted by the worker thread.
	HANDLE hIOCPort = mib->hIOCPort;
	SERVICE_STATUS_HANDLE hSS = mib->hServiceStatusHandle;
	SERVICE_STATUS ss = mib->serviceStatus;

	/* create worker thread */
	HANDLE hWorkerThread = CreateThread(NULL, 0,
			(LPTHREAD_START_ROUTINE)ServiceWorker, mib, 0, NULL);

	if (service) {
		DWORD dwCurrState = SERVICE_RUNNING;

		do {
			DWORD dwBytesTransferred;
			DWORD dwControlCode;
			LPOVERLAPPED po;

			GetQueuedCompletionStatus(hIOCPort, &dwBytesTransferred,
					&dwControlCode, &po, INFINITE);

			switch(dwControlCode) {
				case SERVICE_CONTROL_PAUSE:
					MasterAgentXMibWin32::runStatus = 1;
					dwCurrState = SERVICE_PAUSED;
					break;

				case SERVICE_CONTROL_STOP:
				case SERVICE_CONTROL_SHUTDOWN:
					MasterAgentXMibWin32::runStatus = 0;
					if (dwCurrState == SERVICE_RUNNING
							|| dwCurrState == SERVICE_PAUSED) {
						mib->set_service_status(SERVICE_STOP_PENDING, 1, 8000);
						dwCurrState = SERVICE_STOP_PENDING;
					}
					break;

				case SERVICE_CONTROL_CONTINUE:
					MasterAgentXMibWin32::runStatus = 2;
					dwCurrState = SERVICE_RUNNING;
					break;

				case SERVICE_CONTROL_INTERROGATE:
					mib->set_service_status(dwCurrState);
					break;
			}
		} while (dwCurrState != SERVICE_STOP_PENDING);
	}

	WaitForSingleObject(hWorkerThread, INFINITE);
	delete v3mp;
	delete snmp;	// must free up socket inside snmp before stopping service

	// use local copies because mib object has been already been destroyed
	if (service) {
		ss.dwCurrentState = SERVICE_STOPPED;
		SetServiceStatus(hSS, &ss);
		CloseHandle(hIOCPort);
	} else {
		// close the duplicate handle that we did't use if not a service
		CloseHandle(MasterAgentXMibWin32::hServiceMainThread);
	}
}


DWORD WINAPI ServiceWorker(LPVOID threadParam)
{
	MasterAgentXMibWin32* mib = (MasterAgentXMibWin32 *)threadParam;
	RequestList* reqList = 0;
	AgentXMaster* agentx = 0;

	if (mib->service)
		mib->set_service_status(SERVICE_START_PENDING, 0, 1000);

	reqList = new RequestList();
	// register v3MP
	reqList->set_v3mp(mib->v3mp);
	// register requestList for outgoing requests
	mib->set_request_list(reqList);
	agentx = new AgentXMaster();
	// we only use TCP on Win32 platform
	agentx->set_connect_mode(AX_USE_TCP_SOCKET);

	mib->set_agentx(agentx);
	// enable auto context creation -> a subagent may register for a
	// context not yet known by the master
	mib->set_auto_context(TRUE);

  	mib->add(new snmpGroup());
	mib->add(new snmp_target_mib());
	mib->add(new snmp_community_mib());
	mib->add(new snmp_notification_mib());
	mib->add(new notification_log_mib());
	mib->add(new agentpp_agentx_mib());
	mib->add(new agentpp_config_mib());	
	// add non persistent SNMPv3 engine object
	mib->add(new V3SnmpEngine());
	mib->add(new MPDGroup());

	UsmUserTable *uut = new UsmUserTable();
	// add non persistent USM statistics
	mib->add(new UsmStats());
	// add the USM MIB - usm_mib MibGroup is used to
	// make user added entries persistent
	mib->add(new usm_mib(uut));
	reqList->set_snmp(mib->snmp);
	// register VACM
	Vacm* vacm = new Vacm(*mib);
	reqList->set_vacm(vacm);

	// start AgentX master
	mib->init();

	OctetStr* initialAuth = mib->get_initial_passphrase("auth");
	OctetStr* initialPriv = mib->get_initial_passphrase("priv");

	if (initialAuth) {

		uut->addNewRow("MD5",
				       SNMPv3_usmHMACMD5AuthProtocol,
					   SNMPv3_usmNoPrivProtocol,
					   (initialAuth) ? *initialAuth : "MD5UserAuthPassword", "");
	
		uut->addNewRow("SHA",
				       SNMPv3_usmHMACSHAAuthProtocol,
					   SNMPv3_usmNoPrivProtocol,
					  (initialAuth) ? *initialAuth : "SHAUserAuthPassword", "");
		if (initialPriv) {
			uut->addNewRow("MD5DES",
					       SNMPv3_usmHMACMD5AuthProtocol,
					       SNMPv3_usmDESPrivProtocol,
						   (initialAuth) ? *initialAuth : "MD5DESUserAuthPassword",
						   (initialPriv) ? *initialPriv : "MD5DESUserPrivPassword");

			uut->addNewRow("SHADES",
					       SNMPv3_usmHMACSHAAuthProtocol,
						   SNMPv3_usmDESPrivProtocol,
		                   (initialAuth) ? *initialAuth : "SHADESUserAuthPassword",
						  (initialPriv) ? *initialPriv : "SHADESUserPrivPassword");
		}
	}
	if (initialAuth) 
		delete initialAuth;
	if (initialPriv) 
		delete initialPriv;


	// initialize security information
    vacm->addNewContext("");

	// Add new entries to the SecurityToGroupTable.
	// Used to determine the group a given SecurityName belongs to. 
	// User "new" of the USM belongs to newGroup

	vacm->addNewGroup(SecurityModel_USM, "initial", 
		"initial", storageType_volatile);
	vacm->addNewGroup(SecurityModel_USM, "MD5", 
		"initial", storageType_volatile);
	vacm->addNewGroup(SecurityModel_USM, "SHA", 
		"initial", storageType_volatile);
	vacm->addNewGroup(SecurityModel_USM, "MD5DES", 
		"initial", storageType_volatile);
	vacm->addNewGroup(SecurityModel_USM, "SHADES", 
		"initial", storageType_volatile);

	vacm->addNewAccessEntry("initial", "",
		SecurityModel_USM, SecurityLevel_noAuthNoPriv, 
		match_exact,
		"restricted", "", 
		"restricted", storageType_nonVolatile);
	vacm->addNewAccessEntry("initial", "",
		SecurityModel_USM, SecurityLevel_authNoPriv, 
		match_exact,
		"internet", "internet", 
		"internet", storageType_nonVolatile);
	vacm->addNewAccessEntry("initial", "",
		SecurityModel_USM, SecurityLevel_authPriv, 
		match_exact,
		"internet", "internet", 
		"internet", storageType_nonVolatile);

	// Defining Views
	// View "v1ReadView" includes all objects starting with "1.3".
	// If the ith bit of the mask is not set (0), then also all objects
	// which have a different subid at position i are included in the 
	// view.
	// For example: Oid "6.5.4.3.2.1", Mask(binary) 110111 
	//              Then all objects with Oid with "6.5.<?>.3.2.1" 
	//              are included in the view, whereas <?> may be any
	//              natural number.

	vacm->addNewView("v1v2cReadView", 
		"1.3",       
		"",             // Mask "" is same as 0xFFFFFFFFFF...
		view_included,  // alternatively: view_excluded
		storageType_nonVolatile);

	vacm->addNewView("v1v2cWriteView", 
		"1.3",       
		"",             // Mask "" is same as 0xFFFFFFFFFF...
		view_included,  // alternatively: view_excluded
		storageType_nonVolatile);

	vacm->addNewView("v1v2cWriteView", 
		"1.3.6.1.6",       
		"",             // Mask "" is same as 0xFFFFFFFFFF...
		view_excluded,  // alternatively: view_excluded
		storageType_nonVolatile);

	vacm->addNewView("v1v2cWriteView", 
		"1.3.6.1.4",       
		"",             // Mask "" is same as 0xFFFFFFFFFF...
		view_excluded,  // alternatively: view_excluded
		storageType_nonVolatile);

	vacm->addNewView("v1v2cNotifyView", 
		"1.3",       
		"",             // Mask "" is same as 0xFFFFFFFFFF...
		view_included,  // alternatively: view_excluded
		storageType_nonVolatile);

	vacm->addNewView("internet", "1.3.6.1","",
		view_included, storageType_nonVolatile);
	vacm->addNewView("restricted", "1.3.6.1.2.1.1","",
		view_included, storageType_nonVolatile);
	vacm->addNewView("restricted", "1.3.6.1.2.1.11","", 
		view_included, storageType_nonVolatile);
	vacm->addNewView("restricted", "1.3.6.1.6.3.10.2.1","", 
		view_included, storageType_nonVolatile);
	vacm->addNewView("restricted", "1.3.6.1.6.3.11.2.1","",
		view_included, storageType_nonVolatile);
	vacm->addNewView("restricted", "1.3.6.1.6.3.15.1.1","", 
		view_included, storageType_nonVolatile);
	

	Vbx* vbs = 0;
	coldStartOid coldOid;
	NotificationOriginator no;
	mib->notify("", coldOid, vbs, 0);

	// init AgentX source address validation
	if (snmpTargetAddrEntry::instance && 
		snmpTargetAddrExtEntry::instance) { 
		snmpTargetAddrEntry::instance->
			add_entry("AgentXSrcAddr", "1.3.6.1.2.1.100.1.5",
			          OctetStr::from_hex_string("7F 00 00 01 00 00"),
					  "agentX", "null");
		MibTableRow* r =
			snmpTargetAddrExtEntry::instance->
				add_row(Oidx::from_string("AgentXSrcAddr", FALSE));
		snmpTargetAddrExtEntry::instance->
			set_row(r, OctetStr::from_hex_string("FF FF FF FF 00 00"), 1500);
		r = agentppAgentXExtSrcAddrEntry::instance->add_row(Oidx::from_string("localhost"));
		r->get_nth(0)->replace_value(new OctetStr("agentX"));
		r->get_nth(1)->replace_value(new SnmpInt32(rowActive));
	}

	if (mib->service) {
		mib->set_service_status(SERVICE_RUNNING);
	}

	Request* req;
	int lastRunStatus = MasterAgentXMibWin32::runStatus;
	while (MasterAgentXMibWin32::runStatus) {
		if ((mib->service) && (lastRunStatus != MasterAgentXMibWin32::runStatus)) {
			switch (MasterAgentXMibWin32::runStatus) {
				case 1: 
					mib->trapPoller->pause();
					mib->set_service_status(SERVICE_PAUSED);
					while (!mib->get_thread_pool()->is_idle()) {
						Sleep(10);
					}
					mib->save_all();
					break;
				case 2:
					mib->trapPoller->pause();
					mib->set_service_status(SERVICE_RUNNING);
					break;
			}
			lastRunStatus = MasterAgentXMibWin32::runStatus;
		}
		if (MasterAgentXMibWin32::runStatus == 2) {
			req = reqList->receive(2);
			if (req) {
			    mib->process_request(req);
			}	
			else if ((reqList->size() == 0) &&
				 (mib->get_thread_pool()->is_idle())) {
				mib->cleanup();
			}
		}
		else if (MasterAgentXMibWin32::runStatus == 1) { // paused
			Sleep(500);
		}
	}
	Snmp::socket_cleanup();
	mib->delete_thread_pool();
	mib->save_all();
	delete mib;
	delete reqList;
	delete agentx;
	return 0;
}


BOOL APIENTRY DllMain( HANDLE hModule, 
                       DWORD  ul_reason_for_call, 
                       LPVOID lpReserved
					 )
{
	switch (ul_reason_for_call)
	{
	case DLL_PROCESS_ATTACH:
	case DLL_THREAD_ATTACH:
	case DLL_THREAD_DETACH:
	case DLL_PROCESS_DETACH:
		break;
	}
    return TRUE;
}

ExtensionDLL::ExtensionDLL(Mib* m,
						   HINSTANCE dll, 
						   HANDLE evtHandle,
						   PFNSNMPEXTENSIONQUERY fQuery,
				           PFNSNMPEXTENSIONQUERYEX fQueryEx,
				           PFNSNMPEXTENSIONTRAP fTrap,
						   const Oidx& oid, 
						   mib_access access): 
	MibComplexEntry(oid, access)
{
	mib = m;
	hDll = dll;
	eventHandle = evtHandle;
	
	query = fQuery;
	queryEx = fQueryEx;
	trap = fTrap;
}

ExtensionDLL::~ExtensionDLL()
{
	if (eventHandle)
		FreeLibrary(hDll);
}

void ExtensionDLL::copy_oid(const Oidx& in, AsnObjectIdentifier& out)
{
	out.idLength = in.len();
	out.ids = (UINT*)SnmpUtilMemAlloc(sizeof(UINT) * out.idLength);
	for (unsigned int i=0; i<out.idLength; i++) 
		out.ids[i] = in[i];
}

boolean ExtensionDLL::get_vb(const SnmpVarBindList& vbList, 
							 unsigned int i, Vbx& vb)
{
	if (i >= vbList.len)
		return FALSE;
	vb.set_oid(Oid((unsigned long *) vbList.list[i].name.ids, 
				   vbList.list[i].name.idLength));
	AsnAny* any = &vbList.list[i].value;
	switch (any->asnType) {
		//case ASN_INTEGER:
		case ASN_INTEGER32: {
			vb.set_syntax(sNMP_SYNTAX_INT32);
			vb.set_value(SnmpInt32(any->asnValue.number));
			break;
		}
							/*
		case ASN_UNSIGNED32: {
			vb.set_value(SnmpUInt32(any->asnValue.unsigned32));
			break;
		}
		*/
		case ASN_COUNTER64: {
			vb.set_syntax(sNMP_SYNTAX_CNTR64);
			vb.set_value(Counter64(any->asnValue.counter64.HighPart,
				any->asnValue.counter64.LowPart));
			break;
		}
		case ASN_OCTETSTRING: {
			vb.set_syntax(sNMP_SYNTAX_OCTETS);
			vb.set_value(OctetStr(any->asnValue.string.stream,
				any->asnValue.string.length));
			break;
		}
		case ASN_OPAQUE: {
			vb.set_syntax(sNMP_SYNTAX_OPAQUE);
			vb.set_value(OpaqueStr(any->asnValue.string.stream,
				any->asnValue.string.length));
			break;
		}
		case ASN_OBJECTIDENTIFIER: {
			vb.set_syntax(sNMP_SYNTAX_OID);
			vb.set_value(Oid((const unsigned long *) any->asnValue.object.ids,
				any->asnValue.object.idLength));
			break;
		}
		case ASN_IPADDRESS: {
			vb.set_syntax(sNMP_SYNTAX_IPADDR);
			Oid oidIP;
			for (unsigned int i=0; i< any->asnValue.address.length; i++) {
				oidIP += any->asnValue.address.stream[i];
			}
			vb.set_value(IpAddress( oidIP.get_printable()));
			break;
		}
		case ASN_COUNTER32: {
			vb.set_syntax(sNMP_SYNTAX_CNTR32);
			vb.set_value(Counter32(any->asnValue.counter));
			break;
		}
		case ASN_GAUGE32: {
			vb.set_syntax(sNMP_SYNTAX_GAUGE32);
			vb.set_value(Gauge32(any->asnValue.gauge));
			break;
		}
		case ASN_TIMETICKS: {
			vb.set_syntax(sNMP_SYNTAX_TIMETICKS);
			vb.set_value(TimeTicks(any->asnValue.ticks));
			break;
		}
		default:
			return FALSE;
	}
	return TRUE;
}

int ExtensionDLL::copy_vbs(Vbx* vbs, unsigned int sz, 
						   SnmpVarBindList* vbList)
{
	if (!vbList)
		return -1;
	vbList->len = sz;
	if (sz == 0)
		vbList->list = 0;
	else
		vbList->list = (SnmpVarBind*)
			SnmpUtilMemAlloc(sz * sizeof(SnmpVarBind));
	for (unsigned int i=0; i<vbList->len; i++) {
		Oidx oid;
		vbs[i].get_oid(oid);

		vbList->list[i].name.idLength = oid.len();
		vbList->list[i].name.ids = 
			(UINT*) SnmpUtilMemAlloc(vbList->list[i].name.idLength * 
									 sizeof(UINT));
		if (!vbList->list[i].name.ids) {
			return i;
		}
		memcpy(vbList->list[i].name.ids, 
			   oid.oidval()->ptr, 
			   vbList->list[i].name.idLength * sizeof(UINT));

		AsnAny* pAsn = &vbList->list[i].value;

		switch (vbs[i].get_syntax()) {
			case sNMP_SYNTAX_INT32: {
				pAsn->asnType = ASN_INTEGER32;
				vbs[i].get_value(pAsn->asnValue.number);
				break;
			}
			/* undistinguishable from GAUGE32
			case sNMP_SYNTAX_UINT32: {
				pAsn->asnType = ASN_UNSIGNED32;
				vbs[i].get_value(pAsn->asnValue.unsigned32);
				break;
			}
			*/
			case sNMP_SYNTAX_CNTR64: {
				Counter64 ctr;
				vbs[i].get_value(ctr);

				pAsn->asnType = ASN_COUNTER64;
				pAsn->asnValue.counter64.HighPart = ctr.high();
				pAsn->asnValue.counter64.LowPart  = ctr.low();
				break;
			}
			case sNMP_SYNTAX_OPAQUE: {
				OctetStr str;
				vbs[i].get_value(str);

				pAsn->asnType = ASN_OPAQUE;
				pAsn->asnValue.string.length = str.len();
				pAsn->asnValue.string.stream = (BYTE*) SnmpUtilMemAlloc(
//					pAsn->asnValue.string.stream, 
					pAsn->asnValue.string.length * sizeof(BYTE));
				if (pAsn->asnValue.string.stream == 0) {
					return i;
				}
				memcpy(pAsn->asnValue.string.stream, 
					str.data(), 
					pAsn->asnValue.string.length * sizeof(BYTE));
				pAsn->asnValue.string.dynamic = TRUE;
				break;
			}
			case sNMP_SYNTAX_OCTETS: {
				OctetStr str;
				vbs[i].get_value(str);

				pAsn->asnType = ASN_OCTETSTRING;
				pAsn->asnValue.string.length = str.len();
				pAsn->asnValue.string.stream = (BYTE*) SnmpUtilMemAlloc(
//					pAsn->asnValue.string.stream, 
					pAsn->asnValue.string.length * sizeof(BYTE));
				if (pAsn->asnValue.string.stream == 0) {
					return i;
				}
				memcpy(pAsn->asnValue.string.stream, 
					str.data(), 
					pAsn->asnValue.string.length * sizeof(BYTE));
				pAsn->asnValue.string.dynamic = TRUE;
				break;
			}
			case sNMP_SYNTAX_OID: {
				Oid dataOid;
				vbs[i].get_value(dataOid);

				pAsn->asnType = ASN_OBJECTIDENTIFIER;
				pAsn->asnValue.object.idLength = dataOid.len();
				pAsn->asnValue.object.ids = (UINT*) SnmpUtilMemAlloc(
					//pAsn->asnValue.object.ids, 
					pAsn->asnValue.object.idLength * sizeof(UINT));
				if (pAsn->asnValue.object.ids == 0) {
					return i;
				}
				memcpy(pAsn->asnValue.object.ids, 
					dataOid.oidval()->ptr, 
					pAsn->asnValue.object.idLength * sizeof(UINT));
				break;
			}
			case sNMP_SYNTAX_IPADDR: {
				const int IPADDRESS_LEN = 4;

				IpAddress addr;
				vbs[i].get_value(addr);

				pAsn->asnType = ASN_IPADDRESS;
				pAsn->asnValue.address.length = IPADDRESS_LEN;
				pAsn->asnValue.address.stream = (BYTE*) SnmpUtilMemAlloc(
					//pAsn->asnValue.address.stream, 
					pAsn->asnValue.address.length);
				if (pAsn->asnValue.address.stream == 0) {
					return 0;
				}
				for (int k=0; k<IPADDRESS_LEN; k++)
					pAsn->asnValue.address.stream[k] = addr[k];
				pAsn->asnValue.address.dynamic = TRUE;
				break;
			}
			case sNMP_SYNTAX_CNTR32: {
				pAsn->asnType = ASN_COUNTER32;
				vbs[i].get_value(pAsn->asnValue.counter);
				break;
			}
			case sNMP_SYNTAX_GAUGE32: {
				pAsn->asnType = ASN_GAUGE32;
				vbs[i].get_value(pAsn->asnValue.gauge);
				break;
			}
			case sNMP_SYNTAX_TIMETICKS: {
				pAsn->asnType = ASN_TIMETICKS;
				vbs[i].get_value(pAsn->asnValue.ticks);
				break;
			}
			case sNMP_SYNTAX_NULL: {
				pAsn->asnType = ASN_NULL;
				pAsn->asnValue.number = 0;
				break;
			}
			default:
				return i;
		}
	}
	return 0;
}



void ExtensionDLL::free_vbs(SnmpVarBindList* vbList) 
{
	if (vbList) {
		for (unsigned int i=0; i<vbList->len; i++) {
			SnmpUtilOidFree(&vbList->list[i].name);
			SnmpUtilAsnAnyFree(&vbList->list[i].value);
			SnmpUtilVarBindFree(&(vbList->list[i]));
		}
	}
	SnmpUtilVarBindListFree(vbList);
}



ExtensionDLL* ExtensionDLL::init(Mib* mib, const char* dllPath) 
{
	HINSTANCE hDll = LoadLibrary(dllPath);
	if (!hDll) {
		LOG_BEGIN(ERROR_LOG | 1);
		LOG("Unable to load extension agent DLL (path)");
		LOG(dllPath);
		LOG_END;
		return 0;
	}

	PFNSNMPEXTENSIONINIT initFunction =
		(PFNSNMPEXTENSIONINIT)GetProcAddress(hDll, "SnmpExtensionInit");
	
	if (!initFunction) {
		LOG_BEGIN(ERROR_LOG | 1);
		LOG("Could not initialize extension agent DLL (path)");
		LOG(dllPath);
		LOG_END;
		FreeLibrary(hDll);
		return 0;
	}
	
	HANDLE eventHandle = 0;
	AsnObjectIdentifier asnOid;
	BOOL status = initFunction(0, &eventHandle, &asnOid);
	if (eventHandle == INVALID_HANDLE_VALUE)
		eventHandle = 0;
	if (!status) {
		LOG_BEGIN(ERROR_LOG | 1);
		LOG("Call to SnmpExtensionInit failed for extension agent DLL (path)");
		LOG(dllPath);
		LOG_END;
		FreeLibrary(hDll);
		return 0;
	}

	Oidx root((unsigned long *)asnOid.ids, asnOid.idLength);
	
	LOG_BEGIN(EVENT_LOG | 1);
	LOG("Call to SnmpExtensionInit succeeded (oid)");
	LOG(root.get_printable());
	LOG_END;
	
	PFNSNMPEXTENSIONQUERY query = 
		(PFNSNMPEXTENSIONQUERY)GetProcAddress(hDll, "SnmpExtensionQuery");
	PFNSNMPEXTENSIONQUERYEX queryEx = 
		(PFNSNMPEXTENSIONQUERYEX)GetProcAddress(hDll, "SnmpExtensionQueryEx");
	PFNSNMPEXTENSIONTRAP trap = 
		(PFNSNMPEXTENSIONTRAP)GetProcAddress(hDll, "SnmpExtensionTrap");

	return new ExtensionDLL(mib, hDll, eventHandle, 
						    query, queryEx, trap, 
							root, READCREATE);
}

ExtensionDLL* ExtensionDLL::init_ex() 
{
	PFNSNMPEXTENSIONINITEX initExFunction =
		(PFNSNMPEXTENSIONINITEX)GetProcAddress(hDll, "SnmpExtensionInitEx");
	if (!initExFunction)
		return 0;

	AsnObjectIdentifier asnOid;
	BOOL status = initExFunction(&asnOid);
	if (!status) {
		return 0;
	}
	Oidx root((unsigned long *)asnOid.ids, asnOid.idLength);

	LOG_BEGIN(EVENT_LOG | 1);
	LOG("Call to SnmpExtensionInitEx succeeded (oid)");
	LOG(root.get_printable());
	LOG_END;
	return new ExtensionDLL(mib, hDll, 0, query, queryEx, trap, 
		                    root, READCREATE);
}

MibEntry*  ExtensionDLL::clone() 
{
	return new ExtensionDLL(mib, hDll, eventHandle, 
		                    query, queryEx, trap, *key(), READCREATE);
}

Oidx ExtensionDLL::find_succ(const Oidx& oid, Request* req)
{
	SnmpVarBindList vbList;
	AsnInteger32 errorStatus = 0;
	AsnInteger32 errorIndex = 0;

	Vbx vb;
	vb.set_oid(oid);
	copy_vbs(&vb, 1, &vbList);

	BOOL result;
	if (!queryEx) {
		result = query(SNMP_PDU_GETNEXT, &vbList, &errorStatus, &errorIndex);
	}
	else {
		unsigned long tid = 0;
		if (req)
			tid = req->get_transaction_id();
		else 
			tid = mib->get_request_list()->create_transaction_id();
		result = queryEx(SNMP_EXTENSION_GET_NEXT, tid, &vbList, &contextInfo, 
			             &errorStatus, &errorIndex);
	}
	if ((!result) || (errorStatus != 0) || 
		(vbList.list[0].value.asnType == ASN_NULL)) {
        free_vbs(&vbList);
		return Oidx();
    }
	Oidx next((unsigned long*)vbList.list[0].name.ids,
		      vbList.list[0].name.idLength);
    free_vbs(&vbList);
	if ((next >= upperBound) || (next < *key()))
		return Oidx();
	return next;
}

void ExtensionDLL::get_request(Request* req, int ind)
{
	SnmpVarBindList vbList;
	AsnInteger32 errorStatus = 0;
	AsnInteger32 errorIndex = 0;

	Vbx vb(req->get_value(ind));
	copy_vbs(&vb, 1, &vbList);

	BOOL result;
	if (!queryEx) {
		result = query(SNMP_PDU_GET, &vbList, &errorStatus, &errorIndex);
	}
	else {
		unsigned long tid = req->get_transaction_id();
		result = queryEx(SNMP_EXTENSION_GET, tid, &vbList, &contextInfo, 
			             &errorStatus, &errorIndex);
	}
	if (result) {
		if (errorStatus != 0) {
			req->error(ind, errorStatus);
		}
		else {
			get_vb(vbList, 0, vb);
			req->finish(ind, vb);
		}
	}
	else {
		req->error(ind, SNMP_ERROR_GENERAL_VB_ERR);
	}
	free_vbs(&vbList);
}

void ExtensionDLL::get_next_request(Request* req, int ind) 
{
	SnmpVarBindList vbList;
	AsnInteger32 errorStatus = 0;
	AsnInteger32 errorIndex = 0;

	Vbx vb(req->get_value(ind));
	copy_vbs(&vb, 1, &vbList);

	BOOL result;
	if (!queryEx) {
		result = query(SNMP_PDU_GET, &vbList, &errorStatus, &errorIndex);
	}
	else {
		unsigned long tid = req->get_transaction_id();
		result = queryEx(SNMP_EXTENSION_GET, tid, &vbList, &contextInfo, 
			             &errorStatus, &errorIndex);
	}
	if (result) {
		if (errorStatus != 0) {
			SnmpVarBindList vbListNext;
			// We got already the next oid as the requested one
			// but we cannot be sure that the the object is still available
			// so we have to do the extra work of going back one step
			vb.set_oid(req->get_oid(ind).predecessor());
			copy_vbs(&vb, 1, &vbListNext);

			BOOL result;
			errorStatus = 0;
			errorIndex = 0;

			if (!queryEx) {
				result = query(SNMP_PDU_GETNEXT, &vbListNext, &errorStatus, &errorIndex);
			}
			else {
				unsigned long tid = req->get_transaction_id();
				result = queryEx(SNMP_EXTENSION_GET_NEXT , tid, &vbListNext, &contextInfo, 
					             &errorStatus, &errorIndex);
			}
			if (result) {
				if (errorStatus != 0) {
					req->error(ind, errorStatus);
				}
				else {
					get_vb(vbListNext, 0, vb);
					req->finish(ind, vb);
				}
			}
			else {
				req->error(ind, SNMP_ERROR_GENERAL_VB_ERR);
			}
			free_vbs(&vbListNext);
		}
		else {
			get_vb(vbList, 0, vb);
			req->finish(ind, vb);
		}
	}
	else {
		req->error(ind, SNMP_ERROR_GENERAL_VB_ERR);
	}
	free_vbs(&vbList);
}

int ExtensionDLL::commit_set_request(Request* req, int ind)
{
	return process_set_request(req, ind, SNMP_EXTENSION_SET_COMMIT);
}

int ExtensionDLL::prepare_set_request(Request* req, int& ind)
{
	return process_set_request(req, ind, SNMP_EXTENSION_SET_TEST);
}

int ExtensionDLL::process_set_request(Request* req, int& ind, int phase) 
{
	if ((phase != SNMP_EXTENSION_SET_COMMIT) && (!queryEx)) {
		return SNMP_ERROR_SUCCESS;
	}
	SnmpVarBindList vbList; 
	AsnInteger32 errorStatus = 0;
	AsnInteger32 errorIndex = 0;

	unsigned int count = 0;
	unsigned int* ids = new unsigned int[req->subrequests()]; 
	ids[count++] = ind;
	for (unsigned int i=ind+1; i<(unsigned int)req->subrequests(); i++) {
		if (oid.is_root_of(req->get_oid(i))) {
			ids[count++] = i;
		}
	}
	Vbx* vbs = new Vbx[count];
	for (unsigned int i=0; i<count; i++) {
		vbs[i] = req->get_value(ids[i]);
		if (phase == SNMP_EXTENSION_SET_TEST) {
			if (ids[i] != ind) {
				req->set_locked(ids[i], this);
			}
		}
	}
	copy_vbs(vbs, count, &vbList);

	BOOL result;
	unsigned long tid = req->get_transaction_id();
	if ((phase == SNMP_EXTENSION_SET_COMMIT) && (!queryEx)) {
		result = query(SNMP_PDU_SET, &vbList, &errorStatus, &errorIndex);
	}
	else {
		result = queryEx(phase, tid, &vbList, &contextInfo, 
						&errorStatus, &errorIndex);
	}
	if (result) {
		if (errorStatus != 0) {
			int errIndex = ind;
			if ((errorIndex >= 1) && (errorIndex <= (int)count)) {
				errIndex = ids[errorIndex-1];
			}
			req->error(errIndex, errorStatus);
			for (unsigned int i=0; i<count; i++) {
				if (ids[i] != errIndex) {
					req->finish(ids[i]);
				}
			}
			free_vbs(&vbList);
			delete[] vbs;
			delete[] ids;
			switch (phase) {
				case SNMP_EXTENSION_SET_COMMIT:
					return SNMP_ERROR_COMITFAIL;
				case SNMP_EXTENSION_SET_UNDO:
					return SNMP_ERROR_UNDO_FAIL;
				default:
					return errorStatus;
			}
		}
		else {
			Vbx vb;
			for (unsigned int i=0; i<count; i++) {
				switch (phase) {
				case SNMP_EXTENSION_SET_COMMIT:
					req->finish(ids[i]);
					break;
				case SNMP_EXTENSION_SET_TEST:
					req->set_ready(ids[i]);
					break;
				}
				if (ids[i] != ind) {
					req->set_unlocked(ids[i]);
				}
			}
			free_vbs(&vbList);
			delete[] vbs;
			delete[] ids;
			return errorStatus;
		}
	}
	else {
		for (unsigned int i=0; i<count; i++) {
			if (ids[i] != ind) {
				req->finish(ids[i]);
			}
		}
		req->error(ind, SNMP_ERROR_GENERAL_VB_ERR);
		free_vbs(&vbList);
		delete[] vbs;
		delete[] ids;
		return SNMP_ERROR_GENERAL_VB_ERR;
	}
	free_vbs(&vbList);
	delete[] vbs;
	delete[] ids;
	return SNMP_ERROR_SUCCESS;
}

void ExtensionDLL::cleanup_set_request(Request* req, int& ind)
{
	if (!queryEx) 
		return;

	SnmpVarBindList vbList; 
	AsnInteger32 errorStatus = 0;
	AsnInteger32 errorIndex = 0;

	unsigned int count = 0;
	unsigned int* ids = new unsigned int[req->subrequests()]; 
	ids[count++] = ind;
	for (unsigned int i=ind+1; i<(unsigned int)req->subrequests(); i++) {
		if (oid.is_root_of(req->get_oid(i))) {
			ids[count++] = i;
		}
	}
	Vbx* vbs = new Vbx[count];
	for (unsigned int i=0; i<count; i++) {
		vbs[i] = req->get_value(ids[i]);
	}

	copy_vbs(vbs, count, &vbList);
	delete[] vbs;
	BOOL result;
	unsigned long tid = req->get_transaction_id();
	result = queryEx(SNMP_EXTENSION_SET_CLEANUP, tid, &vbList, &contextInfo, 
			            &errorStatus, &errorIndex);
	for (unsigned int i=0; i<count; i++) {
		if (ids[i] != ind) {
			req->set_unlocked(ids[i]);
		}
	}
	delete[] ids;
	free_vbs(&vbList);
}

int ExtensionDLL::undo_set_request(Request* req, int& ind)
{
	return process_set_request(req, ind, SNMP_EXTENSION_SET_UNDO);
}


void TrapPoller::process_trap(ExtensionDLL* dll) 
{
	if (!dll->trap)
		return;
	AsnObjectIdentifier enterprise;
	AsnInteger32 specificID;
	AsnInteger32 genericID;
	AsnTimeticks sysUpTime;
	SnmpVarBindList* vbList = 
		(SnmpVarBindList*)SnmpUtilMemAlloc(sizeof(SnmpVarBindList));
	while ((!stopit) && 
		   ((pauseit) || (dll->trap(&enterprise, &genericID, &specificID, 
									&sysUpTime, vbList)))) {
		if (pauseit) {
			Sleep(500);
		}
		else {
			Oidx trapOid((unsigned long*)enterprise.ids, enterprise.idLength);
			trapOid += 0ul;
			if (genericID < 6) {
				trapOid = "1.3.6.1.6.3.1.1.5";
				trapOid += genericID + 1;
			}
			else {
				trapOid += specificID;
			}
			Vbx* vbs = new Vbx[vbList->len];
			for (unsigned int i=0; i<vbList->len; i++) {
				dll->get_vb(*vbList, i, vbs[i]); 
			}
			if (!stopit)
				mib->notify("", trapOid, vbs, vbList->len, sysUpTime);
			ExtensionDLL::free_vbs(vbList);
			vbList = (SnmpVarBindList*)SnmpUtilMemAlloc(sizeof(SnmpVarBindList));
			delete[] vbs;
		}
	}
	ExtensionDLL::free_vbs(vbList);
}

void TrapPoller::run() 
{
	HANDLE* eventHandles = 0;
	ExtensionDLL** dlls = 0;
	DWORD n=0;
	while (!stopit) {
		lock();
		if (!eventHandles) {
			if (extensions.size() == 0) {
				LOG_BEGIN(WARNING_LOG | 2);
				LOG("No extension agents registered, trap polling disabled");
				LOG_END;
				unlock();
				return;
			}
			eventHandles = new HANDLE[extensions.size()+1];	// plus hStopEvent
			dlls = new ExtensionDLL*[extensions.size()];
			n = 0;
			for (int i=0; i<extensions.size(); i++) {
				ExtensionDLL* dll = (ExtensionDLL*)extensions.getNth(i);
				if (dll->get_event_handle()) {
					eventHandles[n] = dll->get_event_handle();
					dlls[n] = dll;
					n++;
				}
			}
			eventHandles[n] = hStopEvent;
		}
		// wait max. 10 seconds
		DWORD status = WaitForMultipleObjects(n+1, eventHandles, FALSE, 10000);
		if ((!stopit) && (status != WAIT_TIMEOUT)) {
			DWORD dllIndex = status;
			if (dllIndex >= WAIT_ABANDONED_0)
				dllIndex -= WAIT_ABANDONED_0;
			else if (dllIndex >= WAIT_OBJECT_0) 
				dllIndex -= WAIT_OBJECT_0;
			else {
				unlock();
				continue;
			}
			if (dllIndex < n)
				process_trap(dlls[dllIndex]);
			else if (dllIndex != n) {	// not the special hStopEvent
				DWORD lastError = GetLastError();
				if (lastError) {
					LOG_BEGIN(ERROR_LOG | 2);
					LOG("Wait for traps failed (reason)");
					LOG(lastError);
					LOG_END;
				}
			}
		}
		unlock();
	}
	delete[] eventHandles;
}


MasterAgentXMibWin32::MasterAgentXMibWin32(): MasterAgentXMib()
{
	trapPoller = new TrapPoller(this);
	hIOCPort = NULL;
	service = FALSE;
}

MasterAgentXMibWin32::~MasterAgentXMibWin32()
{
	trapPoller->stop();
	delete trapPoller;
}

void MasterAgentXMibWin32::set_service_name(char* serviceName)
{
	if (serviceName) {
		serviceTableEntries[0].lpServiceName = serviceName;
		LOG_BEGIN(INFO_LOG | 0);
		LOG("Set service (name)");
		LOG(serviceName);
		LOG_END;
	}
}

boolean MasterAgentXMibWin32::start_service()
{
	if(!StartServiceCtrlDispatcher(serviceTableEntries)) {
		long nError = GetLastError();
		LOG_BEGIN(ERROR_LOG | 1);
		LOG("StartServiceCtrlDispatcher failed (error)");
		LOG(nError); 
		LOG_END;
		return FALSE;
	}

	/* Wait for ServiceMain thread to complete.  see KB201349. */
	if (MasterAgentXMibWin32::hServiceMainThread) {
		WaitForSingleObject(MasterAgentXMibWin32::hServiceMainThread, INFINITE);
		CloseHandle(MasterAgentXMibWin32::hServiceMainThread);
	}
	return TRUE;
}

boolean MasterAgentXMibWin32::install_service(char* exePath) 
{
	SC_HANDLE scManager = OpenSCManager(NULL, NULL, SC_MANAGER_CREATE_SERVICE); 
	if (scManager == 0) {
		long nError = GetLastError();
		LOG_BEGIN(ERROR_LOG | 1);
		LOG("OpenSCManager failed (error)");
		LOG(nError); 
		LOG_END;
	}
	else {
		SC_HANDLE schService = CreateService
		( 
			scManager,	/* SCManager database      */ 
			serviceTableEntries[0].lpServiceName,			/* name of service         */ 
			serviceTableEntries[0].lpServiceName,			/* service name to display */ 
			SERVICE_ALL_ACCESS,        /* desired access          */ 
			SERVICE_WIN32_OWN_PROCESS|SERVICE_INTERACTIVE_PROCESS , /* service type            */ 
			SERVICE_AUTO_START,      /* start type              */ 
			SERVICE_ERROR_NORMAL,      /* error control type      */ 
			exePath,			/* service's binary        */ 
			NULL,                      /* no load ordering group  */ 
			NULL,                      /* no tag identifier       */ 
			NULL,                      /* no dependencies         */ 
			NULL,                      /* LocalSystem account     */ 
			NULL
		);                     /* no password             */ 
		if (schService == 0) {
			long nError =  GetLastError();
			LOG_BEGIN(ERROR_LOG | 1);
			LOG("Failed to create (service) (error)");
			LOG(serviceTableEntries[0].lpServiceName);
			LOG(nError); 
			LOG_END;
		}
		else {
			LOG_BEGIN(INFO_LOG | 0);
			LOG("Service installed (service)");
			LOG(serviceTableEntries[0].lpServiceName);
			LOG_END;
			CloseServiceHandle(schService); 
			CloseServiceHandle(scManager);
			return TRUE;
		}
		CloseServiceHandle(scManager);
	}	
	return FALSE;
}

boolean MasterAgentXMibWin32::uninstall_service() 
{
	SC_HANDLE scManager = OpenSCManager( NULL, NULL, SC_MANAGER_ALL_ACCESS); 
	if (scManager == 0) {
		LOG_BEGIN(ERROR_LOG | 1);
		LOG("OpenSCManager failed");
		LOG_END;
	}
	else {
		SC_HANDLE schService = OpenService( scManager, serviceTableEntries[0].lpServiceName, DELETE);
		if (schService == 0) {
			long nError = GetLastError();
			LOG_BEGIN(ERROR_LOG | 1);
			LOG("Open service failed (service) (error)");
			LOG(serviceTableEntries[0].lpServiceName);
			LOG(nError); 
			LOG_END;
		}
		else {
			if (!DeleteService(schService)) {
				LOG_BEGIN(ERROR_LOG | 1);
				LOG("Failed to delete service (service)");
				LOG(serviceTableEntries[0].lpServiceName);
				LOG_END;
			}
			else {
				LOG_BEGIN(INFO_LOG | 0);
				LOG("Deleted service (service)");
				LOG(serviceTableEntries[0].lpServiceName);
				LOG_END;
				CloseServiceHandle(schService); 
				CloseServiceHandle(scManager);	
				return TRUE;
			}
			CloseServiceHandle(schService); 
		}
		CloseServiceHandle(scManager);	
	}
	return FALSE;
}

boolean MasterAgentXMibWin32::init()
{
	HKEY hKey, eKey;

    // Open key
    if (RegOpenKeyEx(HKEY_LOCAL_MACHINE,
			TEXT("SYSTEM\\CurrentControlSet\\Services\\SNMP\\Parameters\\ExtensionAgents\\"),
    		0, KEY_READ | KEY_EXECUTE, &hKey)
			== ERROR_SUCCESS) {
        DWORD  dwValues, dwMaxVal, dwMaxValData;

    	// Find out how many values exist
    	if (RegQueryInfoKey(hKey, NULL, NULL, NULL, NULL, NULL, NULL,
        		&dwValues, &dwMaxVal, &dwMaxValData, NULL, NULL)
				== ERROR_SUCCESS) {
    		TCHAR *szValue = new TCHAR[dwMaxVal + 1];
    		TCHAR *extensionDLL = (TCHAR *)(new BYTE[dwMaxValData]);

    		if (szValue && extensionDLL) {
				TCHAR extensionPath[1024];
				TCHAR extensionPathExpanded[1024];

        		// Get all values
        		for (DWORD iValue = 0; iValue < dwValues; iValue++) {
            		DWORD	dwValueLen = dwMaxVal + 1;
            		DWORD   dwNameLen = dwMaxValData;
            		DWORD   dwType;
			DWORD	dwBufLen;

            		// Get name
            		if (RegEnumValue(hKey, iValue, szValue, &dwValueLen,
							NULL, &dwType, (LPBYTE)extensionDLL, &dwNameLen) != ERROR_SUCCESS
            				|| dwType != REG_SZ)
						continue;

                	/* Use the data found in extensionDLL */
			if (RegOpenKeyEx(HKEY_LOCAL_MACHINE, extensionDLL,
					0, KEY_QUERY_VALUE, &eKey) != ERROR_SUCCESS)
				continue;

			dwBufLen = sizeof(extensionPath);
			DWORD dwErr = RegQueryValueEx(eKey, TEXT("Pathname"),
							NULL, NULL, (LPBYTE)extensionPath, &dwBufLen);
			RegCloseKey(eKey);
			if (dwErr == ERROR_SUCCESS) {
				ExpandEnvironmentStrings(extensionPath, extensionPathExpanded,
							sizeof(extensionPathExpanded)/sizeof(TCHAR));
				ExtensionDLL *extDLL = ExtensionDLL::init(this, extensionPathExpanded);
				do {
					if (extDLL) {
						if (!add(extDLL)) {
							LOG_BEGIN(WARNING_LOG | 1);
							LOG("MasterAgentXMibWin32 could not add extension agent (name)(path)(oid)");
							LOG(extensionDLL);
							LOG(extensionPath);
							LOG(extDLL->key()->get_printable());
							LOG_END;
						}
						else {
							LOG_BEGIN(INFO_LOG | 1);
							LOG("MasterAgentXMibWin32 added extension agent (name)(path)(oid)");
							LOG(extensionDLL);
							LOG(extensionPath);
							LOG(extDLL->key()->get_printable());
							LOG_END;

							trapPoller->add_extension(extDLL);
						}
						extDLL = extDLL->init_ex();
					}
				} while (extDLL);
			}
		}
	}

    	// Free buffers
    	delete[] szValue;
    	delete[] extensionDLL;
	}

    	// Close key
        RegCloseKey( hKey );
	}

	init_source_addresses();
	init_communities();
	// read SNMPv1/v2c trap configuration from Windows registry
	init_notification_targets();
	persistent_objects_path = get_config_path();

	boolean status = MasterAgentXMib::init();
	init_trap_poller();
	return status;
}

OctetStr* MasterAgentXMibWin32::get_config_path() const 
{
	HKEY configPathKey;
    char path[1024];
    char pathExpanded[1024];
    DWORD dwBufLen;

	int status = RegOpenKeyEx(HKEY_LOCAL_MACHINE,
				              "SOFTWARE\\Agentpp\\AgentConfig\\ConfigPath\\",
							  0, KEY_READ, &configPathKey);
	if (status == ERROR_SUCCESS) {
		char key[32];
		for (int i = 0; i<1; i++) { 
			DWORD maxKey = sizeof(key);
			dwBufLen = sizeof(path);
			int retCode = RegEnumValue(configPathKey, i, key, &maxKey, NULL, NULL, (LPBYTE)path, &dwBufLen); 
			if (retCode == (DWORD)ERROR_SUCCESS) {
				// expand enivronment variables
				ExpandEnvironmentStrings(path, pathExpanded, 
				                         sizeof(pathExpanded));
				OctetStr* configPath = new OctetStr(pathExpanded);
				LOG_BEGIN(INFO_LOG | 1);
				LOG("MasterAgentXMibWin32: read config path from registry (path)");
				LOG(configPath->get_printable());
				LOG_END;
				return configPath;
			}
		}
	}
	RegCloseKey(configPathKey);
	return 0;

}

OctetStr* MasterAgentXMibWin32::get_initial_passphrase(const OctetStr& type) const 
{
	HKEY hKey;
    char passphrase[1024];
    DWORD dwBufLen;

	int status = RegOpenKeyEx(HKEY_LOCAL_MACHINE,
				              "SOFTWARE\\Agentpp\\AgentConfig\\SNMPv3\\USM\\InitialPassphrase",
							  0, KEY_READ, &hKey);
	if (status == ERROR_SUCCESS) {
		char key[32];
		for (int i = 0, status = ERROR_SUCCESS; 
				    status == ERROR_SUCCESS; i++) { 
			DWORD maxKey = sizeof(key);
			dwBufLen = sizeof(passphrase);
			int retCode = RegEnumValue(hKey, i, key, &maxKey, NULL, NULL, (LPBYTE)passphrase, &dwBufLen); 
			if (retCode == (DWORD)ERROR_SUCCESS) {
				OctetStr match((unsigned char*)key, maxKey);
				if (match == type) {
					return new OctetStr((unsigned char*)passphrase, dwBufLen-1);
				}
			}
		}
	}
	RegCloseKey(hKey);
	return 0;
}


int MasterAgentXMibWin32::create_engine_key(const OctetStr& engineID)
{
	// create a unique value for this engineID
	// assumes that engineID was not found by get_engine_key
	DWORD dwDisposition;
	HKEY hkBootCounts;
	LONG lRes = RegCreateKeyEx(HKEY_LOCAL_MACHINE, 
			"SOFTWARE\\Agentpp\\AgentConfig\\SNMPv3\\EngineID\\", 0, "", 0, 
			KEY_READ | KEY_WRITE, 0, &hkBootCounts, &dwDisposition);
	if (lRes == ERROR_SUCCESS) {
		for (int id = 1; id < 1000; id++) {		// find first ID that is not used
			char ids[4];
			sprintf(ids, "%d", id);
			lRes = RegQueryValueEx(hkBootCounts, ids, NULL, NULL, NULL, NULL);
			if (lRes != ERROR_SUCCESS) {
				lRes = RegSetValueEx(hkBootCounts, ids, NULL, REG_BINARY,
						engineID.data(), engineID.len());
				if (lRes != ERROR_SUCCESS)
					break;
				RegCloseKey(hkBootCounts);
				return id;
			}
		}
		RegCloseKey(hkBootCounts);
	}
	return -1;
}


int MasterAgentXMibWin32::get_engine_key(const OctetStr& engineID) {
	HKEY engineKey;
    char matchEngineID[100];
    DWORD dwBufLen;
	int found = -1;

	int status = RegOpenKeyEx(HKEY_LOCAL_MACHINE,
				              "SOFTWARE\\Agentpp\\AgentConfig\\SNMPv3\\EngineID\\",
							  0, KEY_READ, &engineKey);
	if (status == ERROR_SUCCESS) {
		char key[32];
		for (int i = 0, retCode = ERROR_SUCCESS; 
				    retCode == ERROR_SUCCESS; i++) { 
			DWORD maxKey = sizeof(key);
			dwBufLen = sizeof(matchEngineID);
			retCode = RegEnumValue(engineKey, i, key, &maxKey, NULL, NULL, (LPBYTE)matchEngineID, &dwBufLen); 
			if (retCode == (DWORD)ERROR_SUCCESS) {
				OctetStr id((unsigned char*)matchEngineID, dwBufLen);
				if (engineID == id) {
					found = atoi(key);
					break;
				}
			}
		}
	}
	RegCloseKey(engineKey);
	if (found == -1)		// engine key does not yet exist, try creating it
		found = create_engine_key(engineID);
	return found;
}

int MasterAgentXMibWin32::get_boot_counter(const OctetStr& engineID, unsigned int& bootCount) 
{
	HKEY bootsKey;
    int boots;
    DWORD dwBufLen = 4;

	int id = get_engine_key(engineID);
	if (id >= 0) {
		char key[4];
		sprintf(key, "%d", id);
		LONG status = RegOpenKeyEx(HKEY_LOCAL_MACHINE,
					               "SOFTWARE\\Agentpp\\AgentConfig\\SNMPv3\\EngineBoots\\",
					               0, KEY_QUERY_VALUE, &bootsKey);
		if (status == ERROR_SUCCESS) {
			if (RegQueryValueEx(bootsKey, key, NULL, NULL,
				                (LPBYTE)&boots, &dwBufLen) 
					!= ERROR_SUCCESS) {
				bootCount = 0;
			}
			else {
				bootCount = boots;
				return SNMPv3_OK;
			}
		}
		else 
			bootCount = 0;
		RegCloseKey(bootsKey);
		return SNMPv3_FILEOPEN_ERROR;
	}
	else {
		bootCount = 0;
	}
	return SNMPv3_OK;
}

int MasterAgentXMibWin32::set_boot_counter(const OctetStr& engineID, unsigned int bootCount) 
{
	int id = get_engine_key(engineID);
	if (id < 0) {
		DWORD dwDisposition;
		HKEY hkBootCounts;
		LONG lRes = RegCreateKeyEx(HKEY_LOCAL_MACHINE, 
			                       "SOFTWARE\\Agentpp\\AgentConfig\\SNMPv3\\EngineID\\", 0, "", 0, 
							       KEY_READ | KEY_WRITE, 0, &hkBootCounts, &dwDisposition); 
		DWORD dwNumValues;   // maximum size of key values
		RegQueryInfoKey(hkBootCounts, NULL, NULL, NULL, NULL, NULL, NULL,
						&dwNumValues, NULL, NULL, NULL, NULL);
		id = dwNumValues+1;
	}
	char ids[10];
	sprintf(ids, "%d", id);
	DWORD dwDisposition;
	HKEY bootsKey;
	LONG status = RegCreateKeyEx(HKEY_LOCAL_MACHINE,
		"SOFTWARE\\Agentpp\\AgentConfig\\SNMPv3\\EngineBoots\\", 0, "", 0, 
		KEY_READ | KEY_WRITE, 0, &bootsKey, &dwDisposition);
	if (status == ERROR_SUCCESS) {
		RegSetValueEx(bootsKey, ids, 0, REG_DWORD, (const BYTE*)&bootCount, sizeof(bootCount));
		LOG_BEGIN(INFO_LOG | 4);
		LOG("MasterAgentXMibWin32 stored boot count into registry (engineID)(bootCount)");
		LOG(engineID.get_printable());
		LOG(bootCount);
		LOG_END;
		return SNMPv3_OK;
	}
	LOG_BEGIN(WARNING_LOG | 1);
	LOG("MasterAgentXMibWin32 could store boot count into registry (engineID)(bootCount)");
	LOG(engineID.get_printable());
	LOG(bootCount);
	LOG_END;
	return SNMPv3_FILE_ERROR;
}

void MasterAgentXMibWin32::add_target_params(const OctetStr& community) 
{
	OctetStr tag("v1trap");
	if (snmpTargetParamsEntry::instance->add_entry(community, // row index
			                                       0,    // mpModel 
			                                       1,    // securityModel
			                                       community, // secName
			                                       1)) { // secLevel
	    snmpNotifyEntry::instance->add_entry(community, // row index 
											 tag,  // tag
											 1);   // type (trap)
	}
}

OctetStr MasterAgentXMibWin32::make_target_address(const UdpAddress& addr) const
{
	OctetStr address;
	IpAddress ip(addr);
	for (int i=0; i<4; i++) {
		address += (unsigned char)ip[i];
	}
	address += (addr.get_port() >> 8);
	address += (addr.get_port() & 0x00FF);
	return address;
}

void MasterAgentXMibWin32::add_target(const UdpAddress& addr, const OctetStr& secName) 
{
	OctetStr tag("v1trap");
	snmpTargetAddrEntry::instance->add_entry(UdpAddress(addr).get_printable(),  // row index
											 Oidx("1.3.6.1.6.1.1"),    // UDP domain
											 make_target_address(addr),  // target address
											 tag,                      // tag
											 secName);                   // params entry
}

void MasterAgentXMibWin32::init_source_addresses()
{
	if ((!snmpTargetAddrEntry::instance) || 
		(!snmpTargetParamsEntry::instance) ||
		(!snmpCommunityEntry::instance) ||
		(!snmpTargetAddrExtEntry::instance)) {
		return;
	}
	HKEY hKey;
    LONG status = RegOpenKeyEx(HKEY_LOCAL_MACHINE,
							   "SYSTEM\\CurrentControlSet\\Services\\SNMP\\Parameters\\PermittedManagers\\",
							   0, KEY_READ, &hKey);
	if (status == ERROR_SUCCESS) {
		char name[32];
		char address[256];
		int found = -1;
		for (int i = 0, status = ERROR_SUCCESS; 
				    status == ERROR_SUCCESS; i++) { 
			DWORD maxName = sizeof(name);
			DWORD maxAddress = sizeof(address);
			status = RegEnumValue(hKey, i, name, &maxName, NULL, NULL, (LPBYTE)&address, &maxAddress); 
			OctetStr tag("v1v2cPermittedManagers");
			if (status == (DWORD)ERROR_SUCCESS) {
				UdpAddress addr(address);
				if (snmpTargetParamsEntry::instance->add_entry("null", // row index
							                                   0,    // mpModel 
										                       1,    // securityModel
													           "null", // secName
															   1)) { // secLevel
				}
				snmpTargetAddrEntry::instance->add_entry(address,
			  										     Oidx("1.3.6.1.6.1.1"),    // UDP domain
														 make_target_address(addr),  // target address
														 tag,                      // tag
														 "null");                   // params entry
				MibTableRow* row = snmpTargetAddrExtEntry::instance->find_index(Oidx::from_string(address, FALSE));
				unsigned char mask[6] = { 255, 255, 255, 255, 0, 0 };
				OctetStr addressMask(mask, 6);
				snmpTargetAddrExtEntry::instance->set_row(row, addressMask, MAX_SNMP_PACKET);
				// activate address validation
				get_request_list()->set_address_validation(TRUE);
			}
		}
		RegCloseKey(hKey);
	}
}

void MasterAgentXMibWin32::init_communities() 
{
	if ((!snmpTargetAddrEntry::instance) || 
		(!snmpTargetParamsEntry::instance) ||
		(!snmpCommunityEntry::instance) ||
		(!snmpTargetAddrExtEntry::instance)) {
		return;
	}
	HKEY hKey;
    LONG status = RegOpenKeyEx(HKEY_LOCAL_MACHINE,
							   "SYSTEM\\CurrentControlSet\\Services\\SNMP\\Parameters\\ValidCommunities\\",
							   0, KEY_READ, &hKey);
	if (status == ERROR_SUCCESS) {
		char community[128];
		int found = -1;
		for (int i = 0, retCode = ERROR_SUCCESS; 
				    retCode == ERROR_SUCCESS; i++) { 
			DWORD maxKey = sizeof(community);
			DWORD value = 0;
		    DWORD dwBufLen = sizeof(value);
			retCode = RegEnumValue(hKey, i, community, &maxKey, NULL, NULL, (LPBYTE)&value, &dwBufLen); 
			if (retCode == (DWORD)ERROR_SUCCESS) {
				OctetStr co((unsigned char*)community, maxKey);
				OctetStr group("v1v2c_");
				group += co;
				OctetStr readView;
				OctetStr writeView;
				OctetStr notifyView;
				if (value >= (1 << SNMP_ACCESS_READ_ONLY)) {
					readView = "v1v2cReadView";
				}
				if ((value >= (1 << SNMP_ACCESS_READ_WRITE)) ||
					(value >= (1 << SNMP_ACCESS_READ_CREATE))) {
					writeView = "v1v2cWriteView";
				}
				if (value >= (1 << SNMP_ACCESS_NOTIFY)) {
					notifyView = "v1v2cNotifyView";
				}
				Vacm* vacm = get_request_list()->get_vacm();
		        vacm->addNewGroup(SecurityModel_v2, co, 
				                  group, storageType_volatile);
		        vacm->addNewGroup(SecurityModel_v1, co, 
				                  group, storageType_volatile);
				vacm->addNewAccessEntry(group, "", 
										SecurityModel_v2, 
										SecurityLevel_noAuthNoPriv, 
										match_exact,
										readView, 
										writeView, 
										notifyView, 
										storageType_nonVolatile);				
				vacm->addNewAccessEntry(group, "", 
										SecurityModel_v1, 
										SecurityLevel_noAuthNoPriv, 
										match_exact,
										readView, 
										writeView, 
										notifyView, 
										storageType_nonVolatile);				
				MibTableRow* row = snmpCommunityEntry::instance->add_row(Oidx::from_string(co, FALSE));
				OctetStr tag("v1v2cPermittedManagers");
				snmpCommunityEntry::instance->set_row(row, co, co,
													  get_request_list()->get_v3mp()->get_local_engine_id(),
													  "", tag, 3, 1);
			}
		}
	}
	RegCloseKey(hKey);
}

void MasterAgentXMibWin32::init_notification_targets() 
{
	if ((!snmpTargetAddrEntry::instance) || 
		(!snmpTargetParamsEntry::instance) ||
		(!snmpNotifyEntry::instance)) {
		return;
	}
	HKEY hKey, cKey;
    LONG status = RegOpenKeyEx(HKEY_LOCAL_MACHINE,
							   "SYSTEM\\CurrentControlSet\\Services\\SNMP\\Parameters\\TrapConfiguration\\",
							   0, KEY_READ, &hKey);
	
	if (status == ERROR_SUCCESS) {
		char community[256];
		int found = -1;
		for (int i = 0, retCode = ERROR_SUCCESS; 
				        retCode == ERROR_SUCCESS; i++) { 
			FILETIME lastWriteTime;
			DWORD maxKey = sizeof(community);
			retCode = RegEnumKeyEx(hKey, i, community, &maxKey, NULL, NULL, NULL, &lastWriteTime); 
			if (retCode == (DWORD)ERROR_SUCCESS) {
				OctetStr secName((unsigned char*)community, maxKey);
				OctetStr communityKey("SYSTEM\\CurrentControlSet\\Services\\SNMP\\Parameters\\TrapConfiguration\\");
				communityKey += community;
				communityKey += "\\";
				status = RegOpenKeyEx(HKEY_LOCAL_MACHINE, communityKey.get_printable(), 0, KEY_READ, &cKey);

				// add v1 target params for this community
				add_target_params(secName);

				LONG retCode2 = ERROR_SUCCESS;
				for (int j=0; retCode2 == ERROR_SUCCESS; j++) {
					char address[256];
					DWORD addressSize = 256;
					char addressName[10];
					DWORD addressNameSize = 10;
					retCode2 = RegEnumValue(cKey, j, addressName, &addressNameSize, 
			 	                            NULL, NULL, (LPBYTE)address, &addressSize); 
					if (retCode2 == (DWORD)ERROR_SUCCESS) {
						UdpAddress addr(address);
						addr.set_port(162);
						add_target(addr, secName);

						LOG_BEGIN(INFO_LOG | 2);
						LOG("MasterAgentXMibWin32 added trap target (address)(community)");
						LOG(addr.get_printable());
						LOG(secName.get_printable());
						LOG_END;
					}
				}
				RegCloseKey(cKey);
			}
		}
		RegCloseKey(hKey);
	}
}

void MasterAgentXMibWin32::init_v3_security()
{

}

void MasterAgentXMibWin32::init_trap_poller()
{
	trapPoller->start();
}

#ifdef AGENTPP_NAMESPACE
}
#endif
